//maze generator by Robert Lindner - 2DAE-GAMEDEVE

//includes
#pragma once;
#include <iostream>
#include <vector>
#include <array>
#include <algorithm>
#include <random>
#include <math.h>
#include <chrono>

#include "save_png.h"

using namespace std;

//Foreward declaration
//Classes
struct Cell;
class Maze;
//Functions
void ExportMaze(const char* filename, const Maze& mazeRef, const unsigned squareSize);
bool TestBorder(unsigned x, unsigned y, unsigned size, const Cell &cellRef);
bool StepComp(const Cell &i, const Cell &j);

//-------------------------------------------
// Classes
//-------------------------------------------
class Timer
{
public:
	Timer()
	{
		begin = std::chrono::high_resolution_clock::now();
		end = std::chrono::high_resolution_clock::now();
	}
	void Start()
	{
		begin = std::chrono::high_resolution_clock::now();
	}
	double GetTime()
	{
		end = std::chrono::high_resolution_clock::now();
		return ((double)(chrono::duration_cast<std::chrono::nanoseconds>(end - begin).count()))*1e-9;
	}
private:
	chrono::steady_clock::time_point begin;
	chrono::steady_clock::time_point end;
};

struct Cell
{
	bool up = false
		, right = false
		, down = false
		, left = false;
	unsigned steps = 0;
	bool isAssigned;
	//precalculation
	size_t nbUp = 0;
	size_t nbRight = 0;
	size_t nbDown = 0;
	size_t nbLeft = 0;
};

class Maze
{
public:
	Maze(unsigned width, unsigned height)
		:m_Width(width)
		, m_Height(height)
	{
		m_Data = std::vector<Cell>(width*height, Cell());
		Precalculate();
	}
	//Data access
    unsigned GetWidth() const { return m_Width; };
	unsigned GetHeight() const { return m_Height; };
	std::vector<Cell> GetData() const { return m_Data; }
	//Operations
	unsigned GetMostSteps() const
	{
		return (*std::max_element(m_Data.begin(), m_Data.end(), StepComp)).steps;
	}
	void Generate()
	{
		random_device rd;
		mt19937 genIndex = std::mt19937(rd());
		mt19937 genOrientation = std::mt19937(rd());

		uniform_int_distribution<unsigned> disIndex = std::uniform_int_distribution<unsigned>(0, m_Width * m_Height - 1);
		uniform_int_distribution<unsigned> disOrientation; 

		unsigned idx = disIndex(genIndex);
		m_Data[idx].isAssigned = true;
		m_Elegible.push_back(idx);
		vector<unsigned> availableStartCells;
		array<unsigned, 4> availableNeighbours;
		unsigned size = 1;
		while (size>0)
		{
			//Get available neighbours
			char neighbourSize = GetAvailableNeighbours(availableNeighbours, idx);
			while (neighbourSize > 0)
			{
				//Pick neighbour at random
				disOrientation = std::uniform_int_distribution<unsigned>(0, neighbourSize-1);
				unsigned neighbourIdx = availableNeighbours[disOrientation(genOrientation)];
				//Remove border to neighbour
				if (neighbourIdx == idx - 1)//left
				{
					m_Data[idx].left = true;
					m_Data[neighbourIdx].right = true;
				}
				else if (neighbourIdx == idx + 1)//right
				{
					m_Data[idx].right = true;
					m_Data[neighbourIdx].left = true;
				}
				else if (neighbourIdx == idx - m_Width)//up
				{
					m_Data[idx].up = true;
					m_Data[neighbourIdx].down = true;
				}
				else//down
				{
					m_Data[idx].down = true;
					m_Data[neighbourIdx].up = true;
				}
				//Go to neighbour
				idx = neighbourIdx;
				m_Data[idx].isAssigned = true;
				m_Elegible.push_back(idx);
				//Get available neighbours
				neighbourSize = GetAvailableNeighbours(availableNeighbours, idx);
			}
			//Get next step
			GetAvailableCells(availableStartCells);
			size = availableStartCells.size();
			disIndex = std::uniform_int_distribution<unsigned>(0, size - 1);
			if(size > 0) idx = availableStartCells[disIndex(genIndex)];
		}
	}
	void CalculateSteps()
	{
		unsigned idx = 0;
		m_Data[idx].steps = 0;
		unsigned steps = 1;
		if (m_Data[idx].right)
		{
			SetStepsAndIterate(idx + 1, steps, 3);
		}
		if (m_Data[idx].down)
		{
			SetStepsAndIterate(idx + m_Width, steps, 0);
		}
	}
private:
	void Precalculate()
	{
		size_t size = m_Data.size();
		for (size_t i = 0; i < size; i++)
		{
			if (i < m_Width) m_Data[i].nbUp = string::npos;
			else m_Data[i].nbUp = i - m_Width;

			if (i%m_Width == m_Width - 1) m_Data[i].nbRight = string::npos;
			else m_Data[i].nbRight = i + 1;

			if (i/m_Width == m_Height-1) m_Data[i].nbDown = string::npos;
			else m_Data[i].nbDown = i + m_Width;

			if (i%m_Width == 0) m_Data[i].nbLeft = string::npos;
			else m_Data[i].nbLeft = i - 1;
		}
	}
	void GetAvailableCells(std::vector<unsigned> &indices)
	{
		indices.clear();
		indices.reserve(m_Height*m_Height);

		for (size_t i = 0; i < m_Elegible.size(); i++)
		{
			unsigned idx = m_Elegible[i];
			if (IsCellAvailable(idx))
			{
				indices.push_back(idx);
			}
		}
	}
	bool IsCellAvailable(unsigned idx)
	{
		Cell c = m_Data[idx];
		if(!(c.nbLeft == string::npos))
			if (!(m_Data[c.nbLeft].isAssigned)) return true;
		if (!(c.nbUp == string::npos))
			if (!(m_Data[c.nbUp].isAssigned)) return true;
		if (!(c.nbRight == string::npos))
			if (!(m_Data[c.nbRight].isAssigned)) return true;
		if (!(c.nbDown == string::npos))
			if (!(m_Data[c.nbDown].isAssigned)) return true;
		return false;
	}
	char GetAvailableNeighbours(array<unsigned, 4> &indices, unsigned idx)
	{
		char ret = 0;
		Cell c = m_Data[idx];
		if (!(c.nbLeft == string::npos))
		{
			if (!(m_Data[c.nbLeft].isAssigned))
			{
				indices[0]=c.nbLeft;
				++ret;
			}
		}
		if (!(c.nbUp == string::npos))
		{
			if (!(m_Data[c.nbUp].isAssigned))
			{
				indices[ret] = c.nbUp;
				++ret;
			}
		}
		if (!(c.nbRight == string::npos))
		{
			if (!(m_Data[c.nbRight].isAssigned))
			{
				indices[ret] =c.nbRight;
				++ret;
			}
		}
		if (!(c.nbDown == string::npos))
		{
			if (!(m_Data[c.nbDown].isAssigned))
			{
				indices[ret]=c.nbDown;
				++ret;
			}
		}
		return ret;
	}
	void SetStepsAndIterate(unsigned idx, unsigned steps, unsigned from)
	{
		m_Data[idx].steps = steps;
		steps++;
		Cell c = m_Data[idx];
		if (c.up && !(from == 0)) SetStepsAndIterate(c.nbUp, steps, 2);
		if (c.right && !(from == 1)) SetStepsAndIterate(c.nbRight, steps, 3);
		if (c.down && !(from == 2)) SetStepsAndIterate(c.nbDown, steps, 0);
		if (c.left && !(from == 3)) SetStepsAndIterate(c.nbLeft, steps, 1);
	}
	unsigned m_Width, m_Height;
	std::vector<Cell> m_Data;
	std::vector<unsigned> m_Elegible;
};


//-------------------------------------------
// Functions
//-------------------------------------------
int main()
{
	unsigned width = 320, height = 180;
	const unsigned SQUARE_SIZE = 6;

	cout << "Creating cells... width: " << width << " height: " << height << " cell size: " << SQUARE_SIZE << endl;
	Timer t = Timer();
	t.Start();
	Maze maze = Maze(width, height);
	cout << "Creation time: " << to_string(t.GetTime()) << endl;

	cout << endl << "Generating maze..." << endl;
	t.Start();
	maze.Generate();
	cout << "Generation time: " << to_string(t.GetTime()) << endl;

	cout << endl << "Calculating steps..." << endl;
	t.Start();
	maze.CalculateSteps();
	cout << "Step caclculation time: " << to_string(t.GetTime()) << endl;

	cout << endl << "Exporting image..." << endl;
	t.Start();
	ExportMaze("maze.png", maze, SQUARE_SIZE);
	cout << "Image saved." << endl;
	cout << "Image export time: " << to_string(t.GetTime()) << endl;

	cout << endl <<"Press ENTER to close the program" << endl;
	cin.get();
	return 0;
}

void ExportMaze(const char* filename, const Maze& mazeRef, const unsigned squareSize)
{
	unsigned sizeX = mazeRef.GetWidth() * squareSize;
	unsigned sizeY = mazeRef.GetHeight() * squareSize;
	unsigned maxSteps = mazeRef.GetMostSteps();
	vector<Cell> data = mazeRef.GetData();
	vector<char> pixels = vector<char>(sizeX * sizeY * 3, 0);
	unsigned bmpWidth = sizeX * 3;

	for (size_t i = 0; i < data.size(); i++)
	{
		float reachability = (float)data[i].steps / (float)maxSteps;
		float g = min(1.f, (1.f - reachability) * 2);
		float r = min(1.f, reachability * 2);
		char green = (char)(g*255.f);
		char red = (char)(r*255.f);

		unsigned idxXBase = (i%mazeRef.GetWidth())*squareSize;
		unsigned idxYBase = (i / mazeRef.GetWidth())*squareSize;

		for (size_t x = 0; x < squareSize; x++)
		{
			unsigned idxX = idxXBase + x;
			for (size_t y = 0; y < squareSize; y++)
			{
				unsigned idxY = idxYBase + y;
				unsigned idx = idxY*bmpWidth + idxX * 3;
				if (TestBorder(x, y, squareSize, data[i]))
				{
					pixels[idx + 1] = 0;
					pixels[idx + 2] = 0;
				}
				else
				{
					pixels[idx + 1] = green;
					pixels[idx + 2] = red;
				}
			}
		}
	}

	save_png(filename, pixels.data(), sizeX, sizeY);
}

bool TestBorder(unsigned x, unsigned y, unsigned size, const Cell &cellRef)
{
	if (x == 0)
	{
		if (!cellRef.left) return true;
		else if (y == 0 || y == size - 1) return true;
	}
	else if (x == size - 1)
	{
		if (!cellRef.right) return true;
		else if (y == 0 || y == size - 1) return true;
	}
	if (y == 0)
	{
		if (!cellRef.up) return true;
	}
	else if (y == size - 1)
	{
		if (!cellRef.down) return true;
	}
	return false;
}
bool StepComp(const Cell &i, const Cell &j)
{
	return i.steps<j.steps;
}